function validateAccessCode(code) {
  const validCodes = (process.env.ACCESS_CODES || '').split(',').map(c => c.trim().toLowerCase());
  if (validCodes.length === 0 || (validCodes.length === 1 && validCodes[0] === '')) {
    return true;
  }
  return validCodes.includes((code || '').trim().toLowerCase());
}

async function deleteTranscript(apiKey, transcriptId) {
  try {
    await fetch(`https://api.assemblyai.com/v2/transcript/${encodeURIComponent(transcriptId)}`, {
      method: 'DELETE',
      headers: { 'Authorization': apiKey },
    });
    console.log(`Transcript deleted: ${transcriptId}`);
  } catch (error) {
    console.error(`Failed to delete transcript ${transcriptId}:`, error.message);
  }
}

export default async function handler(req, res) {
  if (req.method !== 'POST') {
    return res.status(405).json({ error: 'Method not allowed' });
  }

  const apiKey = process.env.ASSEMBLYAI_API_KEY;
  if (!apiKey) {
    return res.status(500).json({ error: 'Service not configured.' });
  }

  // Validate access code
  const accessCode = req.headers['x-access-code'];
  if (!validateAccessCode(accessCode)) {
    return res.status(403).json({ error: 'Invalid access code.' });
  }

  const { transcript_id } = req.body;
  
  // Validate transcript ID format
  if (!transcript_id || typeof transcript_id !== 'string' || !/^[a-zA-Z0-9_-]+$/.test(transcript_id)) {
    return res.status(400).json({ error: 'Invalid request.' });
  }

  const prompt = `You are an expert coach educator providing developmental feedback on a coaching session. Your role is to help this coach grow and become more skilled - this is NOT a pass/fail assessment.

Analyze this coaching conversation using the ICF Core Competencies at the ACC (Associate Certified Coach) level. 

Structure your feedback as follows:

## What You're Doing Well 🌟
Identify 3-4 specific strengths you observed. Be concrete - quote or reference specific moments from the session. Celebrate what's working.

## Opportunities for Growth 🌱
Identify 2-3 areas where the coach could develop further. Frame these as opportunities, not deficits. For each one:
- Describe what you observed
- Explain the relevant ICF competency
- Offer a specific, actionable suggestion for how to strengthen this skill

## Powerful Moments 💫
Highlight 1-2 moments in the session that were particularly effective coaching. Explain what made them work.

## One Thing to Try Next Time
Give one simple, specific thing the coach could experiment with in their next session.

Remember: This coach is learning and growing. Your feedback should be warm, encouraging, and specific. Help them see both their progress and their potential.

ICF Core Competencies to consider:
1. Demonstrates Ethical Practice
2. Embodies a Coaching Mindset
3. Establishes and Maintains Agreements
4. Cultivates Trust and Safety
5. Maintains Presence
6. Listens Actively
7. Evokes Awareness
8. Facilitates Client Growth`;

  try {
    const response = await fetch('https://api.assemblyai.com/lemur/v3/generate/task', {
      method: 'POST',
      headers: {
        'Authorization': apiKey,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        transcript_ids: [transcript_id],
        prompt: prompt,
        final_model: 'anthropic/claude-3-5-sonnet',
      }),
    });

    const data = await response.json();
    
    if (!response.ok) {
      console.error('LeMUR error:', data);
      return res.status(response.status).json({ error: 'Analysis failed.' });
    }

    console.log(`Analysis complete for: ${transcript_id}, time: ${new Date().toISOString()}`);

    // Auto-delete transcript from AssemblyAI for privacy (fire and forget)
    deleteTranscript(apiKey, transcript_id);

    return res.status(200).json({ response: data.response });
  } catch (error) {
    console.error('Analysis error:', error.message);
    return res.status(500).json({ error: 'Analysis failed.' });
  }
}
