function validateAccessCode(code) {
  const validCodes = (process.env.ACCESS_CODES || '').split(',').map(c => c.trim().toLowerCase());
  if (validCodes.length === 0 || (validCodes.length === 1 && validCodes[0] === '')) {
    return true;
  }
  return validCodes.includes((code || '').trim().toLowerCase());
}

export default async function handler(req, res) {
  if (req.method !== 'GET') {
    return res.status(405).json({ error: 'Method not allowed' });
  }

  const apiKey = process.env.ASSEMBLYAI_API_KEY;
  if (!apiKey) {
    return res.status(500).json({ error: 'Service not configured.' });
  }

  // Validate access code
  const accessCode = req.headers['x-access-code'];
  if (!validateAccessCode(accessCode)) {
    return res.status(403).json({ error: 'Invalid access code.' });
  }

  const { id } = req.query;
  
  // Validate transcript ID format (alphanumeric)
  if (!id || typeof id !== 'string' || !/^[a-zA-Z0-9_-]+$/.test(id)) {
    return res.status(400).json({ error: 'Invalid request.' });
  }

  try {
    const response = await fetch(`https://api.assemblyai.com/v2/transcript/${encodeURIComponent(id)}`, {
      headers: {
        'Authorization': apiKey,
      },
    });

    const data = await response.json();
    
    if (!response.ok) {
      return res.status(response.status).json({ error: 'Status check failed.' });
    }

    // Only return necessary fields (don't leak full response structure)
    return res.status(200).json({
      id: data.id,
      status: data.status,
      error: data.error || null,
      utterances: data.utterances || null,
      text: data.text || null,
    });
  } catch (error) {
    console.error('Status error:', error.message);
    return res.status(500).json({ error: 'Status check failed.' });
  }
}
