export const config = {
  api: {
    bodyParser: false,
  },
};

// Simple in-memory rate limiting (resets on cold start, but good enough for basic protection)
const uploadTracker = new Map();
const MAX_UPLOADS_PER_HOUR = 5;
const MAX_FILE_SIZE = 500 * 1024 * 1024; // 500MB limit

function validateAccessCode(code) {
  const validCodes = (process.env.ACCESS_CODES || '').split(',').map(c => c.trim().toLowerCase());
  if (validCodes.length === 0 || (validCodes.length === 1 && validCodes[0] === '')) {
    // No access codes configured = open access (for testing)
    return true;
  }
  return validCodes.includes((code || '').trim().toLowerCase());
}

function checkRateLimit(clientId) {
  const now = Date.now();
  const hourAgo = now - (60 * 60 * 1000);
  
  // Clean old entries
  const uploads = (uploadTracker.get(clientId) || []).filter(time => time > hourAgo);
  
  if (uploads.length >= MAX_UPLOADS_PER_HOUR) {
    return false;
  }
  
  uploads.push(now);
  uploadTracker.set(clientId, uploads);
  return true;
}

function getClientId(req) {
  // Use IP + user agent as a simple client identifier
  const forwarded = req.headers['x-forwarded-for'];
  const ip = forwarded ? forwarded.split(',')[0].trim() : req.socket?.remoteAddress || 'unknown';
  return ip;
}

export default async function handler(req, res) {
  if (req.method !== 'POST') {
    return res.status(405).json({ error: 'Method not allowed' });
  }

  const apiKey = process.env.ASSEMBLYAI_API_KEY;
  if (!apiKey) {
    console.error('ASSEMBLYAI_API_KEY not configured');
    return res.status(500).json({ error: 'Service not configured. Please contact administrator.' });
  }

  // Validate access code from header
  const accessCode = req.headers['x-access-code'];
  if (!validateAccessCode(accessCode)) {
    return res.status(403).json({ error: 'Invalid access code. Please check with your instructor.' });
  }

  // Rate limiting
  const clientId = getClientId(req);
  if (!checkRateLimit(clientId)) {
    return res.status(429).json({ error: 'Upload limit reached. Please try again later.' });
  }

  try {
    // Collect the raw body with size limit
    const chunks = [];
    let totalSize = 0;
    
    for await (const chunk of req) {
      totalSize += chunk.length;
      if (totalSize > MAX_FILE_SIZE) {
        return res.status(413).json({ error: 'File too large. Maximum size is 500MB.' });
      }
      chunks.push(chunk);
    }
    
    const body = Buffer.concat(chunks);
    
    if (body.length === 0) {
      return res.status(400).json({ error: 'No file provided.' });
    }

    // Log upload (without sensitive data)
    console.log(`Upload: ${clientId}, size: ${(body.length / 1024 / 1024).toFixed(2)}MB, time: ${new Date().toISOString()}`);

    // Forward to AssemblyAI
    const response = await fetch('https://api.assemblyai.com/v2/upload', {
      method: 'POST',
      headers: {
        'Authorization': apiKey,
        'Content-Type': 'application/octet-stream',
      },
      body: body,
    });

    const data = await response.json();
    
    if (!response.ok) {
      console.error('AssemblyAI upload error:', data);
      return res.status(response.status).json({ error: 'Upload failed. Please try again.' });
    }

    return res.status(200).json(data);
  } catch (error) {
    console.error('Upload error:', error.message);
    return res.status(500).json({ error: 'Upload failed. Please try again.' });
  }
}
